$input v_normal, v_tangent, v_bitangent, v_view, v_texcoord0

/*
 * Copyright 2016 Jean-François Verdon. All rights reserved.
 * License: http://www.opensource.org/licenses/BSD-2-Clause
 */

/*
 * Shader inspiration:
 *  * BGFX's IBL example:   https://github.com/bkaradzic/bgfx/blob/master/examples/18-ibl/fs_ibl_mesh.sc
 *  * glTF's sample viewer: https://github.com/KhronosGroup/glTF-Sample-Viewer/blob/master/src/shaders/metallic-roughness.frag
 */

#include "common.sh"

uniform vec4 u_camera_position;
uniform vec4 u_pbr_params;
uniform vec4 u_color;

#define u_metallicFactor  u_pbr_params.x
#define u_roughnessFactor u_pbr_params.y
#define u_exposure        u_pbr_params.z

SAMPLERCUBE(s_diffuseEnvironmentSampler, 0);
SAMPLERCUBE(s_specularEnvironmentSampler, 1);

#ifdef MATERIAL_HAS_COLOR_MAP
SAMPLER2D(s_colorSampler, 2);
#endif

#ifdef MATERIAL_HAS_NORMAL_MAP
SAMPLER2D(s_normalSampler, 3);
#endif

struct MaterialInfo
{
  float perceptualRoughness;    // roughness value, as authored by the model creator (input to shader)
  vec3 diffuseColor;            // color contribution from diffuse lighting
  vec3 specularColor;           // color contribution from specular lighting
};

vec3 getIBLContribution(MaterialInfo materialInfo, vec3 n, vec3 v)
{
  float NdotV = clamp(dot(n, v), 0.0, 1.0);

  // Note: Environment textures are filtered with cmft: https://github.com/dariomanesku/cmft
  // Params used:
  // --excludeBase true //!< First level mip is not filtered.
  // --mipCount 7       //!< 7 mip levels are used in total, [256x256 .. 4x4]. Lower res mip maps should be avoided.
  // --glossScale 10    //!< Spec power scale. See: specPwr().
  // --glossBias 2      //!< Spec power bias. See: specPwr().
  float lod = clamp(materialInfo.perceptualRoughness * float(6.0), 0.0, float(6.0)); // Use mip levels [1..6] for radiance.

  vec2 brdfSamplePoint = clamp(vec2(NdotV, materialInfo.perceptualRoughness), vec2(0.0, 0.0), vec2(1.0, 1.0));
  //TODO vec2 brdf = texture2D(u_brdfLUT, brdfSamplePoint).rg;

  vec4 diffuseSample = textureCube(s_diffuseEnvironmentSampler, n);

  vec3 reflection = normalize(reflect(-v, n));
  vec4 specularSample = textureCubeLod(s_specularEnvironmentSampler, reflection, lod);

  vec3 diffuseLight = toLinear(diffuseSample).rgb;
  vec3 specularLight = toLinear(specularSample).rgb;

  vec3 diffuse = diffuseLight * materialInfo.diffuseColor;
  //TODO vec3 specular = specularLight * (materialInfo.specularColor * brdf.x + brdf.y);
  vec3 specular = specularLight * (materialInfo.specularColor);

  return diffuse + specular;
}

void main()
{
#ifdef MATERIAL_HAS_COLOR_MAP
  vec4 baseColor = toLinear(texture2D(s_colorSampler, v_texcoord0)) * u_color;
#else
  vec4 baseColor = u_color;
#endif
  baseColor.a = 1.0; // AlphaMode == OPAQUE

#ifdef MATERIAL_IS_UNLIT
  gl_FragColor = vec4(toGamma(baseColor.rgb), baseColor.a);
  return;
#endif

  vec3 f0 = vec3_splat(0.04);
  vec3 diffuseColor = baseColor.rgb * (vec3_splat(1.0) - f0) * (1.0 - u_metallicFactor);
  vec3 specularColor = mix(f0, baseColor.rgb, u_metallicFactor);

  // Roughness is authored as perceptual roughness; as is convention,
  // convert to material roughness by squaring the perceptual roughness [2].
  float alphaRoughness = u_roughnessFactor * u_roughnessFactor;

  // Compute reflectance.
  float reflectance = max(max(specularColor.r, specularColor.g), specularColor.b);

  vec3 specularEnvironmentR0 = specularColor.rgb;
  // Anything less than 2% is physically impossible and is instead considered to be shadowing. Compare to "Real-Time-Rendering" 4th editon on page 325.
  vec3 specularEnvironmentR90 = vec3(clamp(reflectance * 50.0, 0.0, 1.0));

  MaterialInfo materialInfo = MaterialInfo(u_roughnessFactor,
                                           diffuseColor,
                                           specularColor);

#ifdef MATERIAL_HAS_NORMAL_MAP
    float normalScale = 1.0;
    vec3 normal = texture2D(s_normalSampler, v_texcoord0).xyz;
    normal = normalize(normal * 2.0 - 1.0);

    // HACK for the test image! https://blender.stackexchange.com/questions/100017/directx-and-opengl-normal-maps !!!
    // Blender uses OpenGL convention for coordinates and normal maps (R:+x,G:+y,B:+z data), and export them as such.
    // BUT bgfx expects DirectX left handed convention, so we need to invert y (normals maps are stored with R:+x,G:-y,B:+z data).
    normal = vec3(normal.x, -normal.y, normal.z);

    mat3 tbn = mat3(v_tangent, v_bitangent, v_normal);
    normal = normalize(tbn * (normal * vec3(normalScale, normalScale, 1.0)));
#else
  vec3 normal = v_normal;
#endif

  vec3 color  = vec3(0.0, 0.0, 0.0);
  vec3 view   = normalize(v_view);

  // IBL
  color += getIBLContribution(materialInfo, normal, view);

  // Exposure
  color = color * exp2(u_exposure);

  // Final color
  gl_FragColor = vec4(toGamma(color), baseColor.a);

  // Debug rendering
  /*
  gl_FragColor = vec4(toGamma(baseColor.rgb), baseColor.a);         // Debug unlit
  gl_FragColor = vec4(toGamma(vec3_splat(u_metallicFactor)), 1.0);  // Debug metallic factor
  gl_FragColor = vec4(toGamma(vec3_splat(u_roughnessFactor)), 1.0); // Debug roughness factor
  gl_FragColor = vec4(toGamma(diffuseColor), 1.0);                  // Debug diffuse color
  gl_FragColor = vec4(toGamma(specularColor), 1.0);                 // Debug specular color
  gl_FragColor = vec4(toGamma(vec3_splat(reflectance)), 1.0);       // Debug reflectance
  gl_FragColor = vec4(normal, 1.0);                                 // Debug normals
  gl_FragColor = vec4(normal.xzy, 1.0);                             // Debug normals, same colors as Blender.
  gl_FragColor = vec4(view, 1.0);                                   // Debug view
  */
}
